#Make maps of city income relative to national per capita personal income over time

library(dplyr)
library(ggplot2)
library(Hmisc)
library(scales)
library(reshape2)
library(ggthemes)
library(maps)

setwd('~/projects/regionineq/')

#Going to loop over MSAs vs CZs
varnames = c('msa','czone')
filenames = c('msa','cz')
cleannames = c('msaname','czname')
citynames = c('CBSA','Commuting Zone')

for(i in 1:length(varnames)){  
  vname = varnames[i]
  fname = filenames[i]
  cname = cleannames[i]
  cityname = citynames[i]
    
  #Make directories
  dir.create(paste('output/bea/',fname,'/maps',sep = ''))
  dir.create(paste('output/bea/',fname,'/maps/categorical',sep = ''))
  dir.create(paste('output/bea/',fname,'/maps/continuous',sep = ''))
  
  #Bring in cz data and national data
  cdta = read.csv(paste('output/bea/dta/',fname,'_gdppc.csv',sep = ''))
  cdta$city = cdta[,vname]
  cdta$cityname = cdta[,cname]
  
  #If MSA, drop one MSA in New Mexico with no population in early years (because county was created in 1981)
  if(fname == 'msa') cdta = filter(cdta, city != '24380')  
  
  #If CZ, drop two CZs in Alaska that don't have data in early years
  if(fname == 'cz') cdta = filter(cdta, city != '34101', city != '34107')
  stopifnot(!is.na(cdta$gdppc))
  
  #Turn inccat into properly ordered factor
  cdta$inccat = factor(cdta$inccat,
                       levels = c('<80%','80-90%','90-110%','110-120%','>120%'))
   
  #Crosswalk from MSAs to counties in the maps
  msas = read.csv('data/mable/cntymsakey.csv')
  msas$reg = msas$polyname
  msas$city = msas[,vname]
  
  cntys = map_data('county')
  cntys$reg = paste(cntys$region, cntys$subregion, sep = ',')
  cntys = merge(cntys,msas,by = 'reg')


#Loop through data and map out continuous 
for(yr in min(cdta$year):max(cdta$year)){
  dta = filter(cdta, year == yr)
  
  #Make maps
  mapdat = merge(cntys,dta,by = 'city')
  mapdat = mapdat %>% arrange(order)  
  
  #Mean income
  pdf(paste('output/bea/',fname,'/maps/continuous/map_cont_gdppc_',yr,'.pdf',sep = ''),height = 4.5)
  gplot = ggplot(data = mapdat, aes(long, lat, group = group)) +
    geom_polygon( aes(fill = gdprel)) +
    coord_map("polyconic") +
    scale_fill_gradient(low = 'white', high = 'black', limits = c(0,2), labels = percent, name = 'Per capita income\nrelative to nation')+
    theme_map() +
    theme(legend.position = 'bottom',legend.justification = 'center',plot.title = element_text(hjust = 0.5)) 
    # ggtitle(paste(cityname,'per capita personal income\nrelative to nation,',yr))
  print(gplot)
  dev.off()
  
  #Mean income in buckets
  pdf(paste('output/bea/',fname,'/maps/categorical/map_cat_gdppc_',yr,'.pdf',sep = ''),height = 4.5)
  gplot = ggplot(data = mapdat, aes(long, lat, group = group)) +
    geom_polygon( aes(fill = inccat)) +
    coord_map("polyconic") +
    scale_fill_grey(start = .9,end = .1,name ='Per capita income\nrelative to nation') +
    theme_map() +
    theme(legend.position = 'bottom',legend.justification = 'center',plot.title = element_text(hjust = 0.5)) +
    guides(fill = guide_legend(ncol = 7)) 
    # ggtitle(paste(cityname,'per capita personal income\nrelative to nation,',yr))
  print(gplot)
  dev.off()
  
}

}
