# Make maps of fiscal capacity ratio with high and low value properties superimposed
# For Detroit, Honolulu, Des Moines
# Note that this program relies on proprietary block-level property data not included in the replication package

library(tidyverse)
library(sf)
library(scales)
library(Hmisc)
library(rnaturalearth)
library(tigris)


dir.create('output/maps_fcr_properties')

bigunit = 'cbsa'
smallunit = 'loc2'
vtype = 'tot'

# Bring in shapefiles
natshp = st_read(paste0('intermediate_data/fiscal_units/shapefiles_national/shp_',bigunit,'_',smallunit,'_national/shp_',bigunit,'_',smallunit,'_national.shp'))
natshp = st_transform(natshp,crs = "EPSG:4326")

# Get background
land = ne_download(type = 'land', scale = 'large',category = 'physical') %>% st_transform(crs = 'EPSG:4326')
lakes = ne_download(type = 'lakes', scale = 'large',category = 'physical') %>% st_transform(crs = 'EPSG:4326')

# Get roads
bigroads = primary_roads() %>% st_transform(crs = 'EPSG:4326')


# Shapefiles have state abbreviations in the name. Bring in a lookup table
stabbr = read.csv('nonproprietary_data/states/stabbrev.csv')
stabbr$fips = str_pad(stabbr$fips,2,pad = '0')

# Limit to only state + PR since that's what the data are for
stabbr = filter(stabbr, !(fips %in% c('78','66','60'))) %>% arrange(fips)


## Get CBSA lookup for blocks and counties
cbsas = read.csv('nonproprietary_data/msa_crosswalks/list1_2020.csv')

# Clean up CBSA
cbsas = mutate(cbsas, st = str_pad(FIPS.State.Code, 2,pad = '0'),
               cnty = str_pad(FIPS.County.Code, 3, pad = '0'),cbsa = str_pad(CBSA.Code,5,pad = '0'),
               csa = str_pad(CSA.Code, 3,pad = '0'),
               micro = Metropolitan.Micropolitan.Statistical.Area == 'Micropolitan Statistical Area') %>% select(st,cnty,cbsa,csa,micro)



# Get TFQ data for adding statistics
tfqdta = read_csv(paste0('output/tfq/tfq_',bigunit, '_', smallunit,'.csv'))

# Bring in fiscal unit data
fudta = read_csv(paste0('output/tfq/tfq_fu/fu_big_',bigunit,'_',smallunit,'_',vtype,'.csv'))
fudta = select(fudta, st,bigint,fuint,clpop,valpc,bigpc)

if(bigunit == 'cbsa') fudta$bigint = str_pad(fudta$bigint,5,pad = '0')
if(bigunit == 'csa') fudta$bigint = str_pad(fudta$bigint,3,pad = '0')
if(bigunit == 'cz') fudta$bigint = str_pad(fudta$bigint,5,pad = '0')

# # Get residential and total per capita as fraction of CBSA
fudta$valpc_pct = fudta$valpc / fudta$bigpc
fudta$lrel = log((fudta$valpc+1) / (fudta$bigpc + 1),base = 10)


# Cap at 1 for mapping = 1000% or 10%
fudta[fudta$lrel > 1, 'lrel'] = 1
fudta[fudta$lrel < -1, 'lrel'] = -1

# Merge FU to shapes
natdta = left_join(natshp, fudta, by = c('bigunit' = 'bigint','smallunit' = 'fuint','st'))


# CBSAs to make are Detroit, Honolulu, Des Moines
msasint = c('46520','19820','19780') 


allprops = NULL
allprops5 = NULL

for(msaint in msasint){
  
  mnameint = tfqdta %>% filter(bigint == msaint) %>% select(bigname) %>% unlist()
  mnameshortint = tfqdta %>% filter(bigint == msaint) %>% select(bigshortname) %>% unlist()
  states = filter(fudta, bigint == msaint) %>% select(st) %>% unique() %>% unlist()
  
  tfqcbsa = tfqdta %>% filter(bigint == msaint) %>% select(paste0('tfq_val_',vtype)) %>% unlist() %>% round(3)
  
  pointshp = NULL
  
  for(stint in states){
    
    # Get abbreviation
    abbr_int = stabbr %>% filter(fips == stint) %>% select(st) %>% unlist()
    
    # Get counties
    cntyint = filter(cbsas, cbsa == msaint, st == stint) %>% select(cnty) %>% unlist()
    
    # Get shapefile of blocks
    shppath = paste('nonproprietary_data/shapefiles/nhgis0055_shape/nhgis0055_shapefile_tl2020_',stint,'0_block_2020/',abbr_int,'_block_2020.shp',sep = '')
    shp = st_read(shppath)
    shp = filter(shp,COUNTYFP20 %in% cntyint)
    
    pointshp = rbind(pointshp,shp) %>% data.frame() %>% st_as_sf()
    
  }
  
  # Bring in property data with percentiles
  load(paste0('intermediate_data/cbsa_properties/cbsa_prop_',msaint,'.Rdata'))
  propdta = mutate(propdta, pct_val = as.numeric(as.character(pct_val)))
  
  # Get count of pctile by block
  pctcnt = propdta %>% ungroup() %>% group_by(GEOID) %>% summarise(
    resp1 = sum(pct_val == 1 & !is.na(pct_val)),
    resp100 = sum(pct_val == 100 & !is.na(pct_val)),
    resp5 = sum(pct_val <= 5 & !is.na(pct_val)),
    resp95 = sum(pct_val >= 96  & !is.na(pct_val)))
  
  
  # Merge onto block shape
  pointshp = left_join(pointshp, pctcnt, by = c('GEOID20' = 'GEOID'))
  
  # In Honolulu there are about 6 properties in P1 on blocks that don't get merged. In Chicago 3 properties
  sum(pointshp$resp1, na.rm = T)
  sum(pctcnt$resp1, na.rm = T)
  
  ## This is the part that takes a long time
  # Get points - low
  samppoints = filter(pointshp, !is.na(resp1), resp1 > 0)
  proppoints_lo = st_sample(samppoints, samppoints$resp1)
  ppl = st_sf(prop = proppoints_lo, ptype = 'Lowest-value 1% of properties',bigunit = msaint)

  # Get points - high
  samppoints = filter(pointshp, !is.na(resp100), resp100 > 0)
  proppoints_hi = st_sample(samppoints, samppoints$resp100)
  pph = st_sf(prop = proppoints_hi, ptype = 'Highest-value 1% of properties',bigunit = msaint)

  # Make combined for legend
  proppoints_all = rbind(ppl,pph, deparse.level = 0)

  proppoints_all = st_transform(proppoints_all, crs = 'EPSG:4326')

  allprops = rbind(allprops,proppoints_all) %>% data.frame() %>% st_as_sf()

}


# Now can loop back and make the maps
for(msaint in msasint){
  
  proppoints_all = filter(allprops, bigunit == msaint)
  mnameshortint = tfqdta %>% filter(bigint == msaint) %>% select(bigshortname) %>% unlist()
  
  # Get fiscal unit frame for just this place
  fuint = filter(natdta, bigunit == msaint) 

  # Get limits
  gbbox = st_bbox(fuint)
  
  if(msaint == '46520') {
    gbbox = c('xmin' = -158.30183946836303, 'ymin' = 21.232481024215144,'xmax' = -157.5808344244214, 'ymax' = 21.737715217161828)
  }
  
  
  if(msaint == '19820') {
    gbbox = c('xmin' = -83.85538080014004, 'ymin' = 42.0869379333699,'xmax' = -82.71536172764647, 'ymax' = 42.91039932204269)
    
  }
  
  if(msaint == '19780'){
    gbbox = c('xmin' = -94.3203142389046, 'ymin' = 41.16770079614573,'xmax' = -93.16999449237025, 'ymax' = 41.890571934742994)
    
  }
  
 
  # Now make map
  pdf(paste0('output/maps_fcr_properties/map_paper_',msaint,'_',mnameshortint,'_1.pdf'),width = 10,height = 7)
  
  if(msaint != '46520') {
  
  gplt = ggplot() + 
    geom_sf(data = land, fill = 'lightgrey') + #,color = NA
    geom_sf(data = lakes, fill = 'aliceblue') + #,color = NA
    geom_sf(data = fuint, aes(fill = lrel), size = 1, color = 'black') +
    geom_sf(data = proppoints_all, aes(color = ptype), size = .25,  alpha = .6, pch = 16) + 
    geom_sf(data = bigroads, color = 'gray40') +
    scale_color_discrete(type = c('#1b6c35','#680285'),name = 'Each dot represents 1 property') +  # black  238b45 175c2e
    scale_fill_gradient2(low = '#a50026', mid = '#f7f7f7',high = '#313695', midpoint = 0,limits = c(-1,1),na.value = 'grey40',
                         name = 'Fiscal capacity ratio\n(appraised value per capita/\nCBSA appraised value per capita)',
                         breaks = c(-1,-0.4771213,0,0.4771213,1),labels = c('Less than 10%','33%','100%','300%','More than 1000%')) +

    theme(
      panel.background = element_rect(fill = 'aliceblue'),
      panel.border = element_rect(linewidth = 1,fill = NA),
      panel.grid = element_blank(),
      axis.ticks = element_blank(),
      axis.text = element_blank(),
      legend.key = element_rect(fill = NA, color = 'black',linewidth = .25),
      plot.title = element_text(hjust = 0)
    ) +
    guides(colour = guide_legend(override.aes = list(size=1.5,alpha = .9))) +
    coord_sf(xlim = c(gbbox['xmin'],gbbox['xmax']),ylim = c(gbbox['ymin'],gbbox['ymax'])) 

  print(gplt)
  dev.off()
  
  }
  
  if(msaint == '46520'){
    
    gplt = ggplot() + 
      geom_sf(data = lakes, fill = 'aliceblue') + #,color = NA
      geom_sf(data = fuint, aes(fill = lrel), size = 1, color = 'black') +
      geom_sf(data = proppoints_all, aes(color = ptype), size = .25,  alpha = .6, pch = 16) + 
      geom_sf(data = bigroads, color = 'gray40') +
      scale_color_discrete(type = c('#1b6c35','#680285'),name = 'Each dot represents 1 property') +  # black  238b45 175c2e
      scale_fill_gradient2(low = '#a50026', mid = '#f7f7f7',high = '#313695', midpoint = 0,limits = c(-1,1),na.value = 'grey40',
                           name = 'Fiscal capacity ratio\n(appraised value per capita/\nCBSA appraised value per capita)',
                           breaks = c(-1,-0.4771213,0,0.4771213,1),labels = c('Less than 10%','33%','100%','300%','More than 1000%')) +
      
      theme(
        panel.background = element_rect(fill = 'aliceblue'),
        panel.border = element_rect(linewidth = 1,fill = NA),
        panel.grid = element_blank(),
        axis.ticks = element_blank(),
        axis.text = element_blank(),
        legend.key = element_rect(fill = NA, color = 'black',linewidth = .25),
        plot.title = element_text(hjust = 0)
      ) +
      guides(colour = guide_legend(override.aes = list(size=1.5,alpha = .9))) +
      coord_sf(xlim = c(gbbox['xmin'],gbbox['xmax']),ylim = c(gbbox['ymin'],gbbox['ymax'])) 

    print(gplt)
    dev.off()
    
  }
}

