# Make maps of the fiscal capacity ratio for all CBSAs
# (Not shown in paper)

library(tidyverse); options(tibble.width = Inf)
library(sf)
library(scales)


dir.create('output/metro_maps')

# Loop through big and small units 
for(bigunit in c('cbsa','cz','csa')){ 
  
  for(smallunit in c('loc2','loc','school')) { #,
    
    dir.create(paste0('output/metro_maps/maps_',bigunit,'_',smallunit))
    
    # Bring in shapefiles
    natshp = st_read(paste0('intermediate_data/fiscal_units/shapefiles_national/shp_',bigunit,'_',smallunit,'_national/shp_',bigunit,'_',smallunit,'_national.shp'))
    
    # Get TFQ data for adding statistics
    tfqdta = read_csv(paste0('output/tfq/tfq_',bigunit, '_', smallunit,'.csv'))
    
    # Loop through types of value
    for(vtype in c('ag','oth','tot','res')){
    
      # Bring in fiscal unit data
      fudta = read_csv(paste0('output/tfq/tfq_fu/fu_big_',bigunit,'_',smallunit,'_',vtype,'.csv'))
      fudta = select(fudta, st,bigint,fuint,clpop,valpc,bigpc)
      
      if(bigunit == 'cbsa') fudta$bigint = str_pad(fudta$bigint,5,pad = '0')
      if(bigunit == 'csa') fudta$bigint = str_pad(fudta$bigint,3,pad = '0')
      if(bigunit == 'cz') fudta$bigint = str_pad(fudta$bigint,5,pad = '0')
      
      # Get residential and total per capita as fraction of CBSA
      fudta$fcr = fudta$valpc / fudta$bigpc
      fudta$lrel = log((fudta$valpc+1) / (fudta$bigpc + 1),base = 10)
      

      # Cap at 1 for mapping = 1000% or 10%
      fudta[fudta$lrel > 1, 'lrel'] = 1
      fudta[fudta$lrel < -1, 'lrel'] = -1

      
      natdta = left_join(natshp, fudta, by = c('bigunit' = 'bigint','smallunit' = 'fuint','st'))

      # Export merged for easier looking later
      dir.create('intermediate_data/fiscal_units/shapefiles_national/shp_nat_merged')
      
      outpath = paste('intermediate_data/fiscal_units/shapefiles_national/shp_nat_merged/shp_',bigunit,'_',smallunit,'_',vtype,'_natmerged.shp',sep = '')
      st_write(natdta,outpath,delete_layer = T)    
  
      bigunits = unique(fudta$bigint)
      
      

    # Loop through all the cbsas
    for(i in 1:length(bigunits)){ 

        bigu = bigunits[i]
        
        popu = tfqdta %>% filter(bigint == bigu) %>% select(paste0('pop_',vtype)) %>% unlist() %>% comma()
        bignameu = tfqdta  %>% filter(bigint == bigu) %>% select(bigname) %>% unlist()
        pdfnameu = str_replace(bignameu,'/','-') 
        tfqu = tfqdta %>% filter(bigint == bigu) %>% select(paste0('tfq_val_',vtype)) %>% unlist() %>% round(3)
        
        # Use shortname for pdf label
        if(bigunit == 'cbsa') pdfnameu = tfqdta  %>% filter(bigint == bigu) %>% select(bigshortname) %>% unlist()
        
        shpd = filter(natdta, bigunit == bigu)

        shpd$lrel = shpd %>% data.frame() %>% select(lrel) %>% unlist()
    
        # Map of relative percent
        pdf(paste0('output/metro_maps/maps_',bigunit,'_',smallunit,'/cmap_lrel_',bigunit,'_',smallunit,'_',bigu,'_',pdfnameu,'_',vtype,'.pdf'),height = 7)
        gplt =  ggplot(shpd, aes(fill = lrel)) +
          geom_sf(size = .1, color = 'black') + # coord_sf(lims_method = 'geometry_bbox',default_crs = NULL) +
          scale_fill_gradient2(low = '#a50026', mid = '#f7f7f7',high = '#313695', midpoint = 0,limits = c(-1,1),na.value = 'grey40',
                               name = 'Fiscal capacity ratio\n(assessed value per capita /\nCBSA assessed value per capita)',
                               breaks = c(-1,-0.69897,-0.30103,0,0.30103,0.69897,1),labels = c('<10%','20%','50%','100%','200%','500%','>1000%')) +
         
          theme(
            panel.background = element_blank(),
            panel.grid = element_blank(),
            axis.ticks = element_blank(),
            axis.text = element_blank(),
            plot.title = element_text(hjust = 0)
          )  +
          ggtitle(paste('Assessed value per capita vs',str_to_upper(bigunit),'mean,',bignameu,str_to_upper(bigunit),'\nTFQ:',tfqu,'\nPopulation:',popu))
        print(gplt)
        dev.off()

    
  }
}
    
  }
}
