# Make map of tax base fragmentation quotient

library(tidyverse)
library(sf)
library(tigris)

dir.create('output/nation_maps')

cntys = counties(cb = T, resolution = '20m', year = 2021) %>% shift_geometry() %>% filter(as.numeric(STATEFP) < 60)
sts = states(cb = T, resolution = '20m') %>% shift_geometry() %>% filter(as.numeric(STATEFP) < 60)

## 2020 CBSAs
cbsa = read.csv('nonproprietary_data/msa_crosswalks/list1_2020.csv')

# Clean up CBSA
cbsa = mutate(cbsa, st = str_pad(FIPS.State.Code, 2,pad = '0'),
              cnty = str_pad(FIPS.County.Code, 3, pad = '0'),cbsa = str_pad(CBSA.Code,5,pad = '0'),
              csa = str_pad(CSA.Code, 3,pad = '0'),
              micro = Metropolitan.Micropolitan.Statistical.Area == 'Micropolitan Statistical Area') %>% select(st,cnty,cbsa,csa,micro)

# Bring in 1990 CZs
czs = read.csv('nonproprietary_data/msa_crosswalks/cty_cz_st_crosswalk.csv')
czs = mutate(czs,
             cz = str_pad(cz, 5,pad = '0'),
             st = str_pad(state_fips,2,pad = '0'),
             fips = str_pad(cty,5,pad = '0'),
             cnty = str_sub(fips,3,5)) %>% select(st,cnty,cz)


## Fix missing counties - cases where county name/fips in 2020 Census
## Differs from that in 1990 CZ data - taking info from https://www.census.gov/programs-surveys/geography/technical-documentation/county-changes.1990.html#list-tab-957819518

cntys = mutate(cntys, czcounty = case_when(
  STATEFP == '08' & COUNTYFP == '014' ~ '013' , #Broomfield county from Boulder County
  STATEFP == '12' & COUNTYFP == '086' ~ '025' ,  # Miami Dade county was just Dade county
  STATEFP == '46' & COUNTYFP == '102' ~ '113', # Oglala Lakota County from Shannon County
  STATEFP == '15' & COUNTYFP == '005' ~ '009', # Treat Kalawao county as part of Maui county since not in CZ crosswalk
  
  # Lots of Alaska changes
  STATEFP == '02' & COUNTYFP == '063' ~ '261',
  STATEFP == '02' & COUNTYFP == '066' ~ '261',
  STATEFP == '02' & COUNTYFP == '068' ~ '290',
  STATEFP == '02' & COUNTYFP == '105' ~ '231',
  STATEFP == '02' & COUNTYFP == '158' ~ '261',
  STATEFP == '02' & COUNTYFP == '195' ~ '280',
  STATEFP == '02' & COUNTYFP == '198' ~ '201',
  STATEFP == '02' & COUNTYFP == '230' ~ '231',
  STATEFP == '02' & COUNTYFP == '275' ~ '280',
  STATEFP == '02' & COUNTYFP == '282' ~ '231', 
  
  T ~ COUNTYFP
  
)
)


cntys = left_join(cntys, cbsa, by = c('STATEFP' = 'st','COUNTYFP' = 'cnty'))
cntys = left_join(cntys, czs, by = c('STATEFP' = 'st','czcounty' = 'cnty'))

cntys = mutate(cntys, 
               cbsa = case_when(
                 !is.na(cbsa) ~ cbsa,
                 is.na(cbsa) ~ STATEFP
               ),
               csa = case_when(
                 !is.na(csa) ~ csa,
                 is.na(csa) ~ STATEFP
               ),
 
               
               )


for(bigunit in c('cbsa','csa','cz')){ 
  
    # Get aggregated shape 
    cntys$bigint = cntys %>% data.frame() %>% select(all_of(bigunit)) %>% unlist()
    mapshp = cntys %>% group_by(bigint) %>% summarise(cnt = length(STATEFP))
    
  for(smallunit in c('loc2','loc','school','stcnty')) { 
    
    # Bring in TFQ
    tfqdta = read_csv(paste0('output/tfq/tfq_',bigunit, '_', smallunit,'.csv'))
    
    for(vtype in c('tot','res')){
      tfqdta$mint = tfqdta[,paste0('tfq_val_',vtype)] %>% unlist()
      tfqdta$mitscat = cut(tfqdta$mint, breaks = c(0,.1,.15,.2,.30,1),include.lowest = T) #,labels = c('0-0.1','0.')
      
      if(bigunit == 'csa') tfqdta$bigint = str_pad(tfqdta$bigint,3,pad ='0')
      if(bigunit %in% c('cz','cbsa')) tfqdta$bigint = str_pad(tfqdta$bigint,5,pad ='0')
     
       mapd = left_join(mapshp, tfqdta, by = 'bigint')
       
       pdf(paste0('output/nation_maps/map_tfq_',bigunit,'_',smallunit,'_',vtype,'.pdf'),width = 12,height = 6)
       gplt = ggplot(mapd , aes(fill = mitscat)) +  
         geom_sf(size = .1, color = 'black') + 
         scale_fill_brewer(name = 'TFQ',
                           palette = 'YlGnBu',direction = 1,na.value = 'grey90',labels = c('0-0.1','0.1-0.15','0.15-0.22','0.2-0.3','>0.3',paste0('Non-', str_to_upper(bigunit))),
                           guide = guide_legend(title.position = 'top',title.hjust = 0)) +
         theme(
           panel.background = element_blank(),
           panel.grid = element_blank(),
           axis.ticks = element_blank(),
           axis.text = element_blank(),
           plot.title = element_text(hjust = .5)
         ) 
       print(gplt)
       
       if(bigunit == 'cbsa' & smallunit == 'loc2' & vtype == 'tot') ggsave('output/nation_maps/fig_5_map_tfq.pdf', plot = gplt)
       if(bigunit == 'cbsa' & smallunit == 'school' & vtype == 'tot') ggsave('output/nation_maps/fig_a311_map_tfq_school.pdf', plot = gplt)
       if(bigunit == 'cbsa' & smallunit == 'loc2' & vtype == 'res') ggsave('output/nation_maps/fig_a321_map_tfq_res.pdf', plot = gplt)
       if(bigunit == 'cbsa' & smallunit == 'loc' & vtype == 'tot') ggsave('output/nation_maps/fig_a331_map_tfq_townships.pdf', plot = gplt)
       if(bigunit == 'cz' & smallunit == 'loc2' & vtype == 'tot') ggsave('output/nation_maps/fig_a341_map_tfq_czs.pdf', plot = gplt)
       if(bigunit == 'csa' & smallunit == 'loc2' & vtype == 'tot') ggsave('output/nation_maps/fig_a351_map_tfq_csas.pdf', plot = gplt)
       dev.off()
       
      
    }
    
  }
}
    